/**
 * Content script: scrape LinkedIn posts from the DOM.
 * CSS selectors based on otomata-tools linkedin.py reference.
 */

function parseMetric(text) {
  if (!text) return 0;
  const m = text.match(/([\d\s.,\u202F\u00A0]+)/);
  if (!m) return 0;
  return parseInt(m[1].replace(/[\s.,\u202F\u00A0]/g, ''), 10) || 0;
}

function parseMetric(text) {
  if (!text) return 0;
  const m = text.match(/([\d\s.,\u202F\u00A0]+)/);
  if (!m) return 0;
  return parseInt(m[1].replace(/[\s.,\u202F\u00A0]/g, ''), 10) || 0;
}

function extractPostData(el) {
  // Content
  let content = '';
  const contentEl = el.querySelector('.break-words') || el.querySelector('.update-components-text');
  if (contentEl) {
    content = contentEl.innerText.trim();
  }
  // Date
  let publishedAt = null;
  const timeEl = el.querySelector('time');
  if (timeEl) {
    publishedAt = timeEl.getAttribute('datetime') || null;
  }

  // Post URL from data-urn
  let url = '';
  const urn = el.getAttribute('data-urn');
  if (urn) {
    const activityId = urn.split(':').pop();
    url = `https://www.linkedin.com/feed/update/urn:li:activity:${activityId}/`;
  }

  // Is repost
  let isRepost = false;
  const header = el.querySelector('.update-components-header__text-view');
  if (header) {
    const headerText = header.innerText.toLowerCase();
    isRepost = headerText.includes('repost') || headerText.includes('a republié');
  }

  // Engagement — scan ALL aria-labels in the post (no dependency on CSS classes)
  // LinkedIn aria-labels are accessibility attributes, more stable than class names.
  // Examples: "33 réactions", "5 commentaires sur le post de X", "2 reposts"
  let reactions = 0;
  let comments = 0;
  let reposts = 0;
  const RX_REACTION = /r[eé]action|j.aime|like/i;
  const RX_COMMENT  = /comment/i;
  const RX_REPOST   = /repost|republi|partage/i;

  for (const btn of el.querySelectorAll('button[aria-label]')) {
    const label = btn.getAttribute('aria-label');
    const num = parseMetric(label);
    if (!num) continue;
    if (RX_REACTION.test(label) && !reactions) reactions = num;
    else if (RX_COMMENT.test(label) && !comments) comments = num;
    else if (RX_REPOST.test(label) && !reposts) reposts = num;
  }

  // Format type detection
  let formatType = 'text';
  if (el.querySelector('.update-components-linkedin-video')) formatType = 'video';
  else if (el.querySelector('.update-components-poll')) formatType = 'poll';
  else if (el.querySelector('.update-components-carousel')) formatType = 'carousel';
  else if (el.querySelector('.update-components-article')) formatType = 'article';
  else if (el.querySelector('.update-components-image')) formatType = 'image';

  return {
    content,
    published_at: publishedAt,
    url,
    format_type: formatType,
    is_repost: isRepost,
    reactions,
    comments,
    reposts,
  };
}

function waitForVisible() {
  if (!document.hidden) return Promise.resolve();
  return new Promise((resolve) => {
    const handler = () => {
      if (!document.hidden) {
        document.removeEventListener('visibilitychange', handler);
        resolve();
      }
    };
    document.addEventListener('visibilitychange', handler);
  });
}

async function autoScroll(targetCount) {
  let lastCount = 0;
  let staleRounds = 0;
  const maxScrolls = targetCount + 10;

  for (let i = 0; i < maxScrolls; i++) {
    // Pause while tab is hidden — scrollBy does nothing in background
    await waitForVisible();

    window.scrollBy(0, 800 + Math.random() * 600);
    await new Promise((r) => setTimeout(r, 600 + Math.random() * 400));

    const items = document.querySelectorAll('.feed-shared-update-v2');

    // Report progress to background
    chrome.runtime.sendMessage({
      type: 'SCRAPE_PROGRESS',
      found: items.length,
      target: targetCount,
      phase: 'scrolling',
    });

    if (items.length >= targetCount) break;

    if (items.length === lastCount) {
      // Only count as stale if the tab is actually visible
      if (!document.hidden) {
        staleRounds++;
        if (staleRounds >= 5) break;
      }
    } else {
      staleRounds = 0;
      lastCount = items.length;
    }
  }
}

async function scrapeLinkedInPosts(maxPosts = 50) {
  await autoScroll(maxPosts);

  const items = document.querySelectorAll('.feed-shared-update-v2');
  const posts = [];

  for (const item of items) {
    if (posts.length >= maxPosts) break;
    const data = extractPostData(item);
    if (data) posts.push(data);
  }

  // Report final extraction count
  chrome.runtime.sendMessage({
    type: 'SCRAPE_PROGRESS',
    found: posts.length,
    target: maxPosts,
    phase: 'extracting',
  });

  return posts;
}

// Message listener
chrome.runtime.onMessage.addListener((msg, sender, sendResponse) => {
  if (msg.type === 'PING') {
    sendResponse({ ok: true });
    return false;
  }

  if (msg.type === 'START_SCRAPE') {
    scrapeLinkedInPosts(msg.maxPosts || 50).then((posts) => {
      sendResponse({ posts, profileUrl: window.location.href });
    });
    return true;
  }

  if (msg.type === 'GET_PAGE_INFO') {
    const isLinkedIn = window.location.hostname === 'www.linkedin.com';
    const isActivityPage = window.location.pathname.includes('/recent-activity/');
    sendResponse({ isLinkedIn, isActivityPage, url: window.location.href });
    return true;
  }
});
